// ELOXP Chrome Extension - Chess.com Content Script
// Detects games and tracks ELOXP on chess.com

(function() {
  'use strict';

  // ============================================
  // State
  // ============================================

  let currentGameId = null;
  let isTracking = false;
  let userColor = null;
  let opponentInfo = null;
  let hudElement = null;
  let lastPgn = '';
  let moveCount = 0;
  let settings = { hudEnabled: true };

  // ============================================
  // Bot Detection Patterns
  // ============================================
  
  const BOT_PATTERNS = [
    /^stockfish/i, /^lichess ai/i, /^ai level/i, /^maia/i, /^leela/i,
    /^komodo/i, /^houdini/i, /^fritz/i, /^rybka/i, /^shredder/i,
    /^crafty/i, /^gnuchess/i, /^fairy-stockfish/i, /bot$/i, /^bot-/i,
    /-bot$/i, /computer/i, /engine/i, /^sf\d+$/i, /^coach-/i,
    /^computer$/i, /^danny$/i, /^hans$/i, /^frankie$/i, /^nelson$/i,
    /^isabel$/i, /^martin$/i, /^antonio$/i, /^fabian$/i, /^wei$/i,
    /^elani$/i, /^li$/i, /^emir$/i, /^scarlett$/i, /^karma$/i
  ];

  function isBot(username) {
    if (!username) return false;
    const name = username.toLowerCase().trim();
    for (const pattern of BOT_PATTERNS) {
      if (pattern.test(name)) return true;
    }
    // Chess.com bots often have specific naming patterns
    if (name.includes('bot') || name.includes('engine') || name.includes('computer')) {
      return true;
    }
    return false;
  }

  // ============================================
  // Utilities
  // ============================================

  function extractGameId() {
    const path = window.location.pathname;
    // Match /game/live/{gameId} or /game/daily/{gameId}
    const liveMatch = path.match(/\/game\/live\/([0-9]+)/);
    const dailyMatch = path.match(/\/game\/daily\/([0-9]+)/);
    return liveMatch?.[1] || dailyMatch?.[1] || null;
  }

  function isGamePage() {
    return window.location.pathname.includes('/game/live/') || 
           window.location.pathname.includes('/game/daily/');
  }

  function isActiveGame() {
    // Check for active game indicators
    return document.querySelector('.clock-component.clock-running') !== null ||
           document.querySelector('.board-layout-main') !== null;
  }

  function getUsernameFromPage() {
    // Try to get logged in username
    const userMenu = document.querySelector('.home-username, .nav-username, [data-username]');
    return userMenu?.textContent?.trim() || userMenu?.getAttribute('data-username') || null;
  }

  function getUserColor() {
    const username = getUsernameFromPage();
    if (!username) return null;

    // Check player positions on the board
    const bottomPlayer = document.querySelector('.board-player-bottom .user-username-component');
    const topPlayer = document.querySelector('.board-player-top .user-username-component');

    if (bottomPlayer?.textContent?.includes(username)) {
      return 'white'; // User on bottom = white
    }
    if (topPlayer?.textContent?.includes(username)) {
      return 'black'; // User on top = black (from white's perspective)
    }

    // Check board orientation
    const board = document.querySelector('.board');
    if (board?.classList.contains('flipped')) {
      return 'black';
    }

    return 'white'; // Default assumption
  }

  function getOpponentInfo() {
    try {
      const username = getUsernameFromPage();
      const players = document.querySelectorAll('.board-player-usernames .user-username-component');
      
      for (const player of players) {
        const playerName = player.textContent?.trim();
        if (playerName && playerName !== username) {
          // Try to get rating
          const ratingEl = player.closest('.board-player-usernames')?.querySelector('.user-rating-component');
          const rating = ratingEl ? parseInt(ratingEl.textContent?.match(/\d+/)?.[0]) : null;
          
          // Check if opponent is a bot
          const isBotAccount = isBot(playerName);
          
          // Also check for Chess.com bot indicator badges
          const botBadge = player.closest('.board-player-usernames')?.querySelector('[data-bot], .bot-badge, .computer-icon');
          
          return {
            username: playerName,
            rating: rating,
            isBot: isBotAccount || !!botBadge
          };
        }
      }
    } catch (e) {
      console.error('Error getting opponent info:', e);
    }
    return { username: 'Opponent', rating: null, isBot: false };
  }

  function getTimeControl() {
    try {
      const timeEl = document.querySelector('.game-time-selector, .game-info-time');
      return timeEl?.textContent?.trim() || null;
    } catch (e) {
      return null;
    }
  }

  function extractPgn() {
    try {
      // Method 1: Try chess.com global object
      if (window.chesscom?.games?.current?.pgn) {
        return window.chesscom.games.current.pgn;
      }

      // Method 2: Try to get from game controller
      const gameController = document.querySelector('[data-game-state]');
      if (gameController) {
        const state = JSON.parse(gameController.getAttribute('data-game-state') || '{}');
        if (state.pgn) return state.pgn;
      }

      // Method 3: Parse from move list
      const moveList = document.querySelector('.move-list, .vertical-move-list');
      if (!moveList) return '';

      const moves = [];
      const moveNodes = moveList.querySelectorAll('.move, .node');
      let currentMoveNum = 1;
      let whiteMoved = false;

      moveNodes.forEach(node => {
        const moveText = node.textContent?.trim();
        // Skip move numbers and non-move content
        if (moveText && !moveText.match(/^\d+\.?$/) && moveText.length <= 10) {
          if (!whiteMoved) {
            moves.push(`${currentMoveNum}. ${moveText}`);
            whiteMoved = true;
          } else {
            moves[moves.length - 1] += ` ${moveText}`;
            whiteMoved = false;
            currentMoveNum++;
          }
        }
      });

      return moves.join(' ');
    } catch (e) {
      console.error('Error extracting PGN:', e);
      return '';
    }
  }

  function countMoves(pgn) {
    if (!pgn) return 0;
    const matches = pgn.match(/\d+\./g);
    return matches ? matches.length : 0;
  }

  function getGameResult() {
    // Check for game over modal
    const gameOverModal = document.querySelector('.game-over-modal, .game-review-modal');
    if (!gameOverModal) return null;

    const resultText = gameOverModal.textContent?.toLowerCase() || '';
    
    // Determine result type
    let resultType = 'unknown';
    if (resultText.includes('checkmate')) resultType = 'checkmate';
    else if (resultText.includes('time')) resultType = 'timeout';
    else if (resultText.includes('resign')) resultType = 'resignation';
    else if (resultText.includes('stalemate')) resultType = 'stalemate';
    else if (resultText.includes('draw') || resultText.includes('agreed')) resultType = 'draw';
    else if (resultText.includes('abandon')) resultType = 'abandonment';

    // Determine winner
    if (resultType === 'draw' || resultType === 'stalemate') {
      return { result: 'draw', resultType };
    }

    // Check for win/loss indicators
    const username = getUsernameFromPage();
    if (resultText.includes('won') || resultText.includes('wins')) {
      // Check if the user won
      if (resultText.includes(username?.toLowerCase() || '')) {
        return { result: 'win', resultType };
      }
      return { result: 'loss', resultType };
    }

    if (resultText.includes('lost') || resultText.includes('loses')) {
      if (resultText.includes(username?.toLowerCase() || '')) {
        return { result: 'loss', resultType };
      }
      return { result: 'win', resultType };
    }

    // Fallback: check result icons/classes
    const winIcon = gameOverModal.querySelector('.icon-font-chess.winner, .result-win');
    const lossIcon = gameOverModal.querySelector('.icon-font-chess.loser, .result-loss');
    
    if (winIcon) return { result: 'win', resultType };
    if (lossIcon) return { result: 'loss', resultType };

    return null;
  }

  // ============================================
  // HUD Management (Same as Lichess)
  // ============================================

  // HUD State
  let sessionEloxp = 0;
  let sessionGames = 0;
  let comboCount = 0;
  let comboTimer = null;
  let lastMultiplierCount = 0; // Track total multipliers to detect new ones
  let shownMultipliers = new Set(); // Track which multipliers have been shown
  let multiplierTotals = {}; // Track cumulative totals to calculate deltas
  
  // Exact ELOXP values for each multiplier (matching multipliers.ts)
  const MULTIPLIER_VALUES = {
    'Pawn Capture!': 100, 'Knight Capture!': 300, 'Bishop Capture!': 300,
    'Rook Capture!': 500, 'Queen Capture!': 900, 'Checkmate!': 1500,
    'Fork Executed!': 300, 'Pin Attack!': 200, 'Skewer!': 200,
    'Discovered Attack!': 400, 'Double Check!': 300, 'Check Delivered!': 100,
    'Castled!': 200, 'En Passant!': 200, 'Pawn Promotion!': 500,
    'Underpromotion!': 700, 'Sacrifice Attack!': 800, 'Center Control!': 50,
    'Space Advantage!': 75, 'Development Bonus!': 200, 'King Safety!': 100,
    'Good Pawn Structure!': 25, 'Passed Pawn!': 300, 'Protected Passed Pawn!': 400,
    'Outpost Knight!': 200, 'Open File Rook!': 100, 'Battery!': 150,
    'Bishop Pair!': 150, '6th/7th Rank Knight!': 250, 'Tempo Gained!': 100,
    'Winning Endgame!': 500, 'Defensive Save!': 600, 'Distant Passed Pawn!': 300,
    'King Domination!': 250, 'Time Pressure Bonus!': 300,
    // Penalties (negative values)
    'Isolated Pawn': -50, 'Doubled Pawns': -100, 'Backward Pawn': -75,
    'Weak Square': -100, 'Hanging Piece!': -200, 'Blunder!': -500,
    'Exposed King!': -200, 'Bad Bishop': -100, 'Shattered Structure': -150,
    'Time-Pressure Blunder!': -300,
  };
  let winStreak = 0;
  let isMuted = false;
  let audioContext = null;
  let isPracticeMode = false;
  let lastShownMultiplier = null;

  // ============================================
  // Sound Effects Engine (Web Audio API)
  // ============================================

  function getAudioContext() {
    if (!audioContext) {
      audioContext = new (window.AudioContext || window.webkitAudioContext)();
    }
    if (audioContext.state === 'suspended') {
      audioContext.resume();
    }
    return audioContext;
  }

  function playSound(soundType, isCritical = false) {
    if (isMuted) return;
    
    try {
      const ctx = getAudioContext();
      const oscillator = ctx.createOscillator();
      const gainNode = ctx.createGain();
      oscillator.connect(gainNode);
      gainNode.connect(ctx.destination);

      const now = ctx.currentTime;
      
      if (isCritical) {
        oscillator.type = 'sawtooth';
        oscillator.frequency.setValueAtTime(220, now);
        oscillator.frequency.exponentialRampToValueAtTime(440, now + 0.1);
        oscillator.frequency.exponentialRampToValueAtTime(880, now + 0.2);
        gainNode.gain.setValueAtTime(0.12, now);
        gainNode.gain.exponentialRampToValueAtTime(0.01, now + 0.4);
        oscillator.start(now);
        oscillator.stop(now + 0.4);

        const osc2 = ctx.createOscillator();
        const gain2 = ctx.createGain();
        osc2.connect(gain2);
        gain2.connect(ctx.destination);
        osc2.type = 'square';
        osc2.frequency.setValueAtTime(330, now);
        osc2.frequency.exponentialRampToValueAtTime(660, now + 0.15);
        gain2.gain.setValueAtTime(0.06, now);
        gain2.gain.exponentialRampToValueAtTime(0.01, now + 0.35);
        osc2.start(now);
        osc2.stop(now + 0.35);
      } else if (soundType === 'tap' || soundType === 'pawn') {
        oscillator.type = 'sine';
        oscillator.frequency.setValueAtTime(800, now);
        oscillator.frequency.exponentialRampToValueAtTime(400, now + 0.08);
        gainNode.gain.setValueAtTime(0.05, now);
        gainNode.gain.exponentialRampToValueAtTime(0.01, now + 0.1);
        oscillator.start(now);
        oscillator.stop(now + 0.1);
      } else if (soundType === 'capture' || soundType === 'slash') {
        oscillator.type = 'sawtooth';
        oscillator.frequency.setValueAtTime(400, now);
        oscillator.frequency.exponentialRampToValueAtTime(200, now + 0.12);
        gainNode.gain.setValueAtTime(0.08, now);
        gainNode.gain.exponentialRampToValueAtTime(0.01, now + 0.15);
        oscillator.start(now);
        oscillator.stop(now + 0.15);
      } else if (soundType === 'ding' || soundType === 'check') {
        oscillator.type = 'sine';
        oscillator.frequency.setValueAtTime(1200, now);
        oscillator.frequency.exponentialRampToValueAtTime(800, now + 0.15);
        gainNode.gain.setValueAtTime(0.07, now);
        gainNode.gain.exponentialRampToValueAtTime(0.01, now + 0.2);
        oscillator.start(now);
        oscillator.stop(now + 0.2);
      } else if (soundType === 'boom' || soundType === 'major') {
        oscillator.type = 'triangle';
        oscillator.frequency.setValueAtTime(150, now);
        oscillator.frequency.exponentialRampToValueAtTime(80, now + 0.2);
        gainNode.gain.setValueAtTime(0.1, now);
        gainNode.gain.exponentialRampToValueAtTime(0.01, now + 0.25);
        oscillator.start(now);
        oscillator.stop(now + 0.25);
      } else if (soundType === 'power' || soundType === 'tactic') {
        oscillator.type = 'triangle';
        oscillator.frequency.setValueAtTime(300, now);
        oscillator.frequency.exponentialRampToValueAtTime(600, now + 0.08);
        oscillator.frequency.exponentialRampToValueAtTime(450, now + 0.15);
        gainNode.gain.setValueAtTime(0.08, now);
        gainNode.gain.exponentialRampToValueAtTime(0.01, now + 0.18);
        oscillator.start(now);
        oscillator.stop(now + 0.18);
      } else {
        oscillator.type = 'triangle';
        oscillator.frequency.setValueAtTime(600, now);
        oscillator.frequency.exponentialRampToValueAtTime(900, now + 0.05);
        oscillator.frequency.exponentialRampToValueAtTime(500, now + 0.12);
        gainNode.gain.setValueAtTime(0.06, now);
        gainNode.gain.exponentialRampToValueAtTime(0.01, now + 0.15);
        oscillator.start(now);
        oscillator.stop(now + 0.15);
      }
    } catch (e) {
      // Audio not supported
    }
  }

  function playComboSound(comboLevel) {
    if (isMuted || comboLevel < 2) return;
    
    try {
      const ctx = getAudioContext();
      const oscillator = ctx.createOscillator();
      const gainNode = ctx.createGain();
      oscillator.connect(gainNode);
      gainNode.connect(ctx.destination);

      const now = ctx.currentTime;
      const baseFreq = 400 + (comboLevel * 80);
      oscillator.type = 'sine';
      oscillator.frequency.setValueAtTime(baseFreq, now);
      oscillator.frequency.exponentialRampToValueAtTime(baseFreq * 1.5, now + 0.1);
      gainNode.gain.setValueAtTime(0.04 + (comboLevel * 0.008), now);
      gainNode.gain.exponentialRampToValueAtTime(0.01, now + 0.15);
      oscillator.start(now);
      oscillator.stop(now + 0.15);
    } catch (e) {}
  }

  function playVictorySound() {
    if (isMuted) return;
    
    try {
      const ctx = getAudioContext();
      const now = ctx.currentTime;
      
      const notes = [523.25, 659.25, 783.99, 1046.50];
      notes.forEach((freq, i) => {
        const osc = ctx.createOscillator();
        const gain = ctx.createGain();
        osc.connect(gain);
        gain.connect(ctx.destination);
        osc.type = 'sine';
        osc.frequency.setValueAtTime(freq, now + i * 0.15);
        gain.gain.setValueAtTime(0.1, now + i * 0.15);
        gain.gain.exponentialRampToValueAtTime(0.01, now + i * 0.15 + 0.3);
        osc.start(now + i * 0.15);
        osc.stop(now + i * 0.15 + 0.35);
      });
    } catch (e) {}
  }

  // SVG Icons
  const ICONS = {
    coin: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polygon points="12 2 15.09 8.26 22 9.27 17 14.14 18.18 21.02 12 17.77 5.82 21.02 7 14.14 2 9.27 8.91 8.26 12 2" fill="#ffff00" stroke="#ffff00"/></svg>`,
    chevronUp: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="18 15 12 9 6 15"></polyline></svg>`,
    flame: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M8.5 14.5A2.5 2.5 0 0 0 11 12c0-1.38-.5-2-1-3-1.072-2.143-.224-4.054 2-6 .5 2.5 2 4.9 4 6.5 2 1.6 3 3.5 3 5.5a7 7 0 1 1-14 0c0-1.153.433-2.294 1-3a2.5 2.5 0 0 0 2.5 2.5z"></path></svg>`,
    sparkles: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="m12 3-1.912 5.813a2 2 0 0 1-1.275 1.275L3 12l5.813 1.912a2 2 0 0 1 1.275 1.275L12 21l1.912-5.813a2 2 0 0 1 1.275-1.275L21 12l-5.813-1.912a2 2 0 0 1-1.275-1.275L12 3Z"></path></svg>`,
    swords: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="14.5 17.5 3 6 3 3 6 3 17.5 14.5"/><line x1="13" y1="19" x2="19" y2="13"/><line x1="16" y1="16" x2="20" y2="20"/><line x1="19" y1="21" x2="21" y2="19"/><polyline points="14.5 6.5 18 3 21 3 21 6 17.5 9.5"/><line x1="5" y1="14" x2="9" y2="18"/><line x1="7" y1="17" x2="4" y2="20"/><line x1="3" y1="19" x2="5" y2="21"/></svg>`,
    zap: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polygon points="13 2 3 14 12 14 11 22 21 10 12 10 13 2"></polygon></svg>`,
    target: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"/><circle cx="12" cy="12" r="6"/><circle cx="12" cy="12" r="2"/></svg>`,
    crown: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="m2 4 3 12h14l3-12-6 7-4-7-4 7-6-7zm3 16h14"></path></svg>`,
    trophy: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M6 9H4.5a2.5 2.5 0 0 1 0-5H6"/><path d="M18 9h1.5a2.5 2.5 0 0 0 0-5H18"/><path d="M4 22h16"/><path d="M10 14.66V17c0 .55-.47.98-.97 1.21C7.85 18.75 7 20.24 7 22"/><path d="M14 14.66V17c0 .55.47.98.97 1.21C16.15 18.75 17 20.24 17 22"/><path d="M18 2H6v7a6 6 0 0 0 12 0V2Z"/></svg>`,
    activity: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="22 12 18 12 15 21 9 3 6 12 2 12"></polyline></svg>`,
    trendingUp: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="22 7 13.5 15.5 8.5 10.5 2 17"/><polyline points="16 7 22 7 22 13"/></svg>`
  };

  function getDivisionClass(division) {
    const d = (division || 'rookie').toLowerCase();
    if (d.includes('grandmaster')) return 'grandmaster';
    if (d.includes('champion')) return 'champion';
    if (d.includes('challenger')) return 'challenger';
    if (d.includes('contender')) return 'contender';
    return 'rookie';
  }

  function createHUD() {
    if (hudElement) return;

    hudElement = document.createElement('div');
    hudElement.id = 'eloxp-hud';
    hudElement.innerHTML = `
      <div class="eloxp-hud-panel">
        <!-- Cyberpunk Corner Brackets -->
        <div class="eloxp-corner-tl"></div>
        <div class="eloxp-corner-tr"></div>
        <div class="eloxp-corner-bl"></div>
        <div class="eloxp-corner-br"></div>
        
        <!-- Scanline Overlay -->
        <div class="eloxp-scanlines"></div>
        
        <!-- Holographic Shimmer -->
        <div class="eloxp-shimmer"></div>

        <!-- Header -->
        <div class="eloxp-hud-header" id="eloxp-hud-header">
          <div class="eloxp-hud-logo">
            <img src="${chrome.runtime.getURL('icons/eloxp-coin-sm.png')}" alt="ELOXP" class="eloxp-coin eloxp-coin-sm" onerror="this.style.display='none'">
            <span>ELOXP</span>
          </div>
          <div class="eloxp-live-indicator" id="eloxp-status-indicator">
            <div class="eloxp-live-dot"></div>
            <span class="eloxp-live-text" id="eloxp-status-text">🤖 PRACTICE</span>
          </div>
          <button class="eloxp-sound-btn" id="eloxp-sound-toggle" title="Toggle Sound">🔊</button>
          <button class="eloxp-toggle-btn" id="eloxp-toggle">▲</button>
        </div>

        <div class="eloxp-hud-body" id="eloxp-hud-body">
          <!-- Division & ELO Progress -->
          <div class="eloxp-division-section">
            <div class="eloxp-division-row">
              <span class="eloxp-division-badge eloxp-division-rookie" id="eloxp-division">ROOKIE</span>
              <span class="eloxp-elo-display" id="eloxp-elo-display">800 ELO</span>
            </div>
            <div class="eloxp-progress-bar">
              <div class="eloxp-progress-fill" id="eloxp-progress-fill" style="width: 0%"></div>
            </div>
            <div class="eloxp-elo-range" id="eloxp-elo-range">800 → 1000</div>
          </div>

          <div class="eloxp-divider"></div>

          <!-- Total Skill Points -->
          <div class="eloxp-total-section">
            <div class="eloxp-total-label">TOTAL XP</div>
            <div class="eloxp-total-value" id="eloxp-count">0</div>
          </div>

          <div class="eloxp-divider"></div>

          <!-- This Bout Section - THE HERO -->
          <div class="eloxp-game-section">
            <div class="eloxp-game-header">
              <span class="eloxp-game-label">THIS BOUT</span>
              <span class="eloxp-move-counter" id="eloxp-move">Round #0</span>
            </div>
            <div class="eloxp-game-xp" id="eloxp-game-xp">+0 XP</div>
            
            <!-- Combo Container - Only shows when active -->
            <div class="eloxp-combo-container" id="eloxp-combo-container"></div>
          </div>

          <!-- Session Summary -->
          <div class="eloxp-session-summary">
            <span class="eloxp-session-label">FIGHT RECORD</span>
            <span id="eloxp-session-record" class="eloxp-session-record">0W-0L</span>
          </div>
        </div>
      </div>

      <!-- Multiplier Popups -->
      <div class="eloxp-popups-container" id="eloxp-popups"></div>
    `;

    document.body.appendChild(hudElement);

    document.getElementById('eloxp-toggle').addEventListener('click', () => {
      hudElement.classList.toggle('eloxp-hud-minimized');
      document.getElementById('eloxp-toggle').textContent = 
        hudElement.classList.contains('eloxp-hud-minimized') ? '▼' : '▲';
    });

    // Sound toggle
    document.getElementById('eloxp-sound-toggle').addEventListener('click', () => {
      isMuted = !isMuted;
      document.getElementById('eloxp-sound-toggle').textContent = isMuted ? '🔇' : '🔊';
      document.getElementById('eloxp-sound-toggle').classList.toggle('eloxp-muted', isMuted);
      if (!isMuted) {
        playSound('tap');
      }
    });

    makeDraggable(hudElement);
    console.log('ELOXP: HUD created');
  }

  // Multiplier emoji icons mapping (matches multipliers.ts exactly)
  const MULTIPLIER_ICONS = {
    // Material Captures
    'pawn': '♙', 'knight capture': '♘', 'bishop capture': '♗', 'rook capture': '♖', 
    'queen capture': '♕', 'checkmate': '♚', 'mate': '♚',
    // Tactics & Specials
    'fork': '🔱', 'pin': '📌', 'skewer': '🗡️', 'discovered': '👁️', 'double check': '✌️',
    'check': '✓', 'castling': '🏰', 'en passant': '↗️', 'promotion': '👸', 
    'underpromotion': '🎭', 'sacrifice': '💥',
    // Positional
    'center': '🎯', 'space': '📐', 'development': '🚀', 'king safety': '🛡️',
    'pawn structure': '🧱', 'passed pawn': '⬆️', 'outpost': '🐴', 'open file': '🗼',
    'battery': '🔋', 'bishop pair': '👯', 'rank knight': '🎖️',
    // Penalties
    'isolated': '🔸', 'doubled': '🔶', 'backward': '⬇️', 'weak square': '🕳️',
    'hanging': '😱', 'blunder': '❌', 'exposed king': '👑💥', 'bad bishop': '🚫♗',
    'shattered': '💔', 'time-pressure': '⏰❌',
    // Endgame & Tempo
    'tempo': '⏱️', 'endgame': '🏆', 'defensive': '🛡️✨', 'distant': '🏃', 'king domination': '👑🎯',
    // Meta
    'upset': '🎉', 'streak': '🔥', 'time pressure bonus': '⚡',
    // Generic
    'strong': '💪', 'good': '👍', 'solid': '✅', 'critical': '🔥', 'eloxp': '🪙'
  };

  function getMultiplierIcon(name) {
    const nameLower = name.toLowerCase();
    for (const [key, icon] of Object.entries(MULTIPLIER_ICONS)) {
      if (nameLower.includes(key)) {
        return icon;
      }
    }
    return '🪙'; // Default ELOXP coin
  }

  function getMultiplierColors(name, points, isCritical) {
    const nameLower = name.toLowerCase();
    
    if (isCritical || points >= 40) return 'eloxp-bg-gold';
    if (nameLower.includes('queen') || nameLower.includes('mate') || nameLower.includes('checkmate')) return 'eloxp-bg-gold';
    if (nameLower.includes('check') || nameLower.includes('fork')) return 'eloxp-bg-yellow';
    if (nameLower.includes('knight') || nameLower.includes('bishop')) return 'eloxp-bg-cyan';
    if (nameLower.includes('rook') || nameLower.includes('castling')) return 'eloxp-bg-blue';
    if (nameLower.includes('pin') || nameLower.includes('skewer')) return 'eloxp-bg-amber';
    if (nameLower.includes('discovered') || nameLower.includes('sacrifice')) return 'eloxp-bg-orange';
    if (nameLower.includes('tempo') || nameLower.includes('center')) return 'eloxp-bg-purple';
    if (nameLower.includes('blunder') || nameLower.includes('hanging')) return 'eloxp-bg-red';
    if (nameLower.includes('mistake')) return 'eloxp-bg-mistake';
    if (points < 0) return 'eloxp-bg-red';
    if (nameLower.includes('streak') || nameLower.includes('critical')) return 'eloxp-bg-pink';
    
    return 'eloxp-bg-green';
  }

  // Add multiplier popup with sound
  function addPopup(name, points, isCritical = false) {
    const container = document.getElementById('eloxp-popups');
    if (!container) return;

    // Get emoji icon and colors from multiplier mapping
    const emojiIcon = getMultiplierIcon(name);
    const bgClass = getMultiplierColors(name, points, isCritical);

    const popup = document.createElement('div');
    const nameLower = name.toLowerCase();
    const isMistake = nameLower.includes('mistake');
    const isBlunder = nameLower.includes('blunder');
    const popupClass = isCritical ? 'eloxp-popup-critical' : (isMistake ? 'eloxp-popup-mistake' : (isBlunder ? 'eloxp-popup-blunder' : ''));
    const pointsClass = points < 0 ? 'eloxp-negative' : '';
    
    popup.className = `eloxp-popup ${popupClass}`;
    popup.innerHTML = `
      <div class="eloxp-popup-bg ${bgClass}"></div>
      ${isCritical ? '<div class="eloxp-popup-shine"></div>' : ''}
      <span class="eloxp-popup-icon ${isCritical ? 'eloxp-animate' : ''}"><span class="eloxp-emoji-icon">${emojiIcon}</span></span>
      <span class="eloxp-popup-name">${name}</span>
      <span class="eloxp-popup-points ${pointsClass}">${points >= 0 ? '+' : ''}${points}</span>
    `;

    while (container.children.length >= 3) {
      container.firstChild.classList.add('eloxp-popup-exit');
      setTimeout(() => container.firstChild?.remove(), 150);
    }

    container.appendChild(popup);

    // 🔊 PLAY SOUND EFFECT based on multiplier type
    if (isCritical) {
      playSound('critical', true);
    } else if (nameLower.includes('queen') || nameLower.includes('checkmate') || nameLower.includes('mate')) {
      playSound('critical', true);
    } else if (nameLower.includes('rook') || nameLower.includes('major')) {
      playSound('boom');
    } else if (nameLower.includes('knight') || nameLower.includes('bishop') || nameLower.includes('capture')) {
      playSound('capture');
    } else if (nameLower.includes('fork') || nameLower.includes('pin') || nameLower.includes('discovered') || nameLower.includes('tactic')) {
      playSound('tactic');
    } else if (nameLower.includes('check')) {
      playSound('ding');
    } else if (nameLower.includes('pawn')) {
      playSound('tap');
    } else if (points >= 20) {
      playSound('power');
    } else if (points >= 10) {
      playSound('capture');
    } else {
      playSound('tap');
    }

    if (isCritical) {
      const flash = document.createElement('div');
      flash.className = 'eloxp-screen-flash eloxp-flash-yellow';
      document.body.appendChild(flash);
      setTimeout(() => flash.remove(), 300);
    }

    setTimeout(() => {
      popup.classList.add('eloxp-popup-exit');
      setTimeout(() => popup.remove(), 150);
    }, 2500);
  }

  // Update combo display with sound and auto-decay
  // Combo display - only shows when active (no placeholder)
  function updateCombo(count) {
    const container = document.getElementById('eloxp-combo-container');
    if (!container) return;

    const previousCount = comboCount;
    comboCount = count;

    // Clear existing timer
    if (comboTimer) {
      clearTimeout(comboTimer);
      comboTimer = null;
    }

    // Set decay timer - combo resets after 8 seconds of no tactical moves
    if (count > 0) {
      comboTimer = setTimeout(() => {
        updateCombo(0);
      }, 8000);
    }

    // 🔊 Play combo sound when combo level increases
    if (count >= 2 && count > previousCount) {
      playComboSound(count);
    }

    if (count < 2) {
      // Hide combo - no confusing placeholder
      container.innerHTML = '';
      container.style.display = 'none';
      hudElement?.querySelector('.eloxp-hud-panel')?.classList.remove('eloxp-hud-combo-active');
    } else {
      // Show exciting combo for streamers!
      container.style.display = 'block';
      const comboClass = count >= 5 ? 'eloxp-combo-mega' : count >= 3 ? 'eloxp-combo-hot' : '';
      const bonusXP = Math.floor(count * 15);
      container.innerHTML = `
        <div class="eloxp-combo-display ${comboClass}">
          <span class="eloxp-combo-label">COMBO</span>
          <span class="eloxp-combo-count">${count}x</span>
          ${count >= 3 ? `<span class="eloxp-combo-bonus">+${bonusXP} XP</span>` : ''}
          ${count >= 4 ? '<span class="eloxp-combo-fire">🔥</span>' : ''}
        </div>
      `;
      hudElement?.querySelector('.eloxp-hud-panel')?.classList.add('eloxp-hud-combo-active');
    }
  }

  function makeDraggable(element) {
    const header = document.getElementById('eloxp-hud-header');
    if (!header) return;
    
    let isDragging = false;
    let startX, startY, startLeft, startTop;

    header.addEventListener('mousedown', (e) => {
      // Don't drag if clicking buttons
      if (e.target.closest('.eloxp-toggle-btn') || e.target.closest('.eloxp-sound-btn')) return;
      
      isDragging = true;
      startX = e.clientX;
      startY = e.clientY;
      const rect = element.getBoundingClientRect();
      startLeft = rect.left;
      startTop = rect.top;
      
      // Disable transitions while dragging
      element.style.setProperty('transition', 'none', 'important');
      header.style.cursor = 'grabbing';
      e.preventDefault();
    });

    document.addEventListener('mousemove', (e) => {
      if (!isDragging) return;
      
      const deltaX = e.clientX - startX;
      const deltaY = e.clientY - startY;
      
      // Use setProperty with 'important' to override CSS rules
      element.style.setProperty('left', `${startLeft + deltaX}px`, 'important');
      element.style.setProperty('top', `${startTop + deltaY}px`, 'important');
      element.style.setProperty('right', 'auto', 'important');
    });

    document.addEventListener('mouseup', () => {
      if (isDragging) {
        isDragging = false;
        element.style.setProperty('transition', '', '');
        header.style.cursor = 'grab';
      }
    });
  }

  function setHUDStatus(status) {
    const indicator = document.getElementById('eloxp-status-indicator');
    const statusText = document.getElementById('eloxp-status-text');
    const dot = indicator?.querySelector('.eloxp-live-dot');
    
    if (!indicator || !statusText || !dot) return;
    
    // Remove any existing status classes
    indicator.className = 'eloxp-live-indicator';
    
    switch (status) {
      case 'live':
        // TRACKING ACTIVE - Green pulsing dot
        statusText.textContent = '🎮 TRACKING';
        dot.style.background = '#00ff00';
        dot.style.boxShadow = '0 0 8px #00ff00';
        indicator.classList.add('tracking-active');
        break;
      case 'practice':
        // Practice mode - vs bots, ELOXP won't count toward Fighter record
        statusText.textContent = '🤖 PRACTICE';
        dot.style.background = '#6b7280';
        dot.style.boxShadow = '0 0 8px #6b7280';
        indicator.classList.add('practice');
        break;
      case 'puzzle':
        statusText.textContent = '🧩 PUZZLE';
        dot.style.background = '#f59e0b';
        dot.style.boxShadow = '0 0 8px #f59e0b';
        break;
      case 'analysis':
        statusText.textContent = '📊 ANALYSIS';
        dot.style.background = '#6366f1';
        dot.style.boxShadow = '0 0 8px #6366f1';
        break;
      case 'waiting':
        statusText.textContent = '⏳ READY';
        dot.style.background = '#94a3b8';
        dot.style.boxShadow = '0 0 8px #94a3b8';
        break;
      case 'game_over':
        // Game ended - show different state
        statusText.textContent = '🏁 COMPLETE';
        dot.style.background = '#ffff00';
        dot.style.boxShadow = '0 0 8px #ffff00';
        indicator.classList.add('game-over');
        break;
      case 'not_tracking':
        // NOT TRACKING - Red indicator
        statusText.textContent = '⚠️ NOT TRACKING';
        dot.style.background = '#ef4444';
        dot.style.boxShadow = '0 0 8px #ef4444';
        indicator.classList.add('not-tracking');
        break;
      default:
        statusText.textContent = '❌ OFF';
        dot.style.background = '#ef4444';
        dot.style.boxShadow = '0 0 8px #ef4444';
    }
  }

  // Check if a multiplier is a tactical move that should increase combo
  function isTacticalMove(name) {
    const tacticalKeywords = [
      'capture', 'check', 'fork', 'pin', 'skewer', 'discovered', 
      'mate', 'sacrifice', 'promotion', 'passant', 'attack',
      'tempo', 'deflection', 'decoy', 'zwischenzug', 'desperado'
    ];
    const nameLower = name.toLowerCase();
    return tacticalKeywords.some(keyword => nameLower.includes(keyword));
  }

  // Division ELO ranges for progress calculation (MUST match main app tiers.ts)
  const DIVISION_RANGES = {
    'rookie': { min: 800, max: 999 },
    'contender': { min: 1000, max: 1499 },
    'challenger': { min: 1500, max: 1999 },
    'champion': { min: 2000, max: 2399 },
    'grandmaster': { min: 2400, max: 3500 }
  };

  function updateHUD(data) {
    if (!hudElement || !settings.hudEnabled) return;

    console.log('ELOXP: Updating HUD with data', data);

    // Update practice mode from API response
    if (data.isPractice !== undefined) {
      isPracticeMode = data.isPractice;
      setHUDStatus(isPracticeMode ? 'practice' : 'live');
    }
    if (data.hudData?.isPractice !== undefined) {
      isPracticeMode = data.hudData.isPractice;
      setHUDStatus(isPracticeMode ? 'practice' : 'live');
    }

    // Update TOTAL XP count with animation
    const countEl = document.getElementById('eloxp-count');
    const totalXP = data.hudData?.totalEloxp || data.totalEloxp || 0;
    const oldValue = parseInt(countEl?.textContent?.replace(/,/g, '')) || 0;
    
    if (countEl && totalXP > 0 && totalXP !== oldValue) {
      countEl.textContent = totalXP.toLocaleString();
      countEl.classList.add('eloxp-animate');
      setTimeout(() => countEl.classList.remove('eloxp-animate'), 400);
    }

    // Update THIS GAME XP - the main hero number
    const currentEloxp = data.currentEloxp || 0;
    if (currentEloxp > 0) {
      sessionEloxp = currentEloxp;
      const gameXpEl = document.getElementById('eloxp-game-xp');
      if (gameXpEl) {
        gameXpEl.textContent = `+${sessionEloxp.toLocaleString()} XP`;
        gameXpEl.classList.add('eloxp-animate');
        setTimeout(() => gameXpEl.classList.remove('eloxp-animate'), 400);
      }
    }
    
    // Show popups for active multipliers - detect NEW multipliers
    // This is the PRIMARY source of popup data from the actual PGN analysis
    let shownMultiplierPopups = false;
    
    if (data.activeMultipliers && data.activeMultipliers.length > 0) {
      for (const multiplier of data.activeMultipliers) {
        const prevTotal = multiplierTotals[multiplier.name] || 0;
        const newTotal = multiplier.total;
        
        // Only show popup if the total increased (new event occurred)
        if (newTotal > prevTotal && multiplier.count > 0) {
          // Calculate the delta (points earned from this specific event)
          const delta = newTotal - prevTotal;
          
          // Use the known per-event ELOXP value if available, otherwise use delta
          const perEventValue = MULTIPLIER_VALUES[multiplier.name] || delta;
          
          // Determine if critical hit (high-value captures, checkmate, queen)
          const isCritical = Math.abs(perEventValue) >= 500 || 
                            multiplier.name.toLowerCase().includes('mate') ||
                            multiplier.name.toLowerCase().includes('queen');
          
          // Show popup with the per-event value, not cumulative
          addPopup(multiplier.name, perEventValue, isCritical);
          shownMultiplierPopups = true;
          
          // Update combo on tactical moves
          if (isTacticalMove(multiplier.name)) {
            comboCount++;
            updateCombo(comboCount);
            
            // Reset combo timer
            if (comboTimer) clearTimeout(comboTimer);
            comboTimer = setTimeout(() => {
              comboCount = 0;
              updateCombo(0);
            }, 8000); // 8 second combo window
          }
        }
        
        // Update tracked total
        multiplierTotals[multiplier.name] = newTotal;
      }
    }

    // Fallback: Only show generic feedback if no specific multipliers were shown
    if (!shownMultiplierPopups && data.eloxpEarned && data.eloxpEarned > 0) {
      const points = data.eloxpEarned;
      
      if (points >= 5) {
        let name = 'Move Played';
        let isCritical = false;

        if (points >= 40) {
          name = 'Critical Hit';
          isCritical = true;
        } else if (points >= 20) {
          name = 'Strong Move';
        } else if (points >= 10) {
          name = 'Good Move';
        } else if (points >= 5) {
          name = 'Solid Play';
        }

        addPopup(name, points, isCritical);
      }
    }

    // Update HUD data (division, ELO, progress)
    if (data.hudData) {
      const division = (data.hudData.division || 'rookie').toLowerCase();
      const elo = data.hudData.elo || data.hudData.eloRating || 800;
      
      // Division badge
      const divisionEl = document.getElementById('eloxp-division');
      if (divisionEl) {
        const divClass = getDivisionClass(division);
        divisionEl.className = `eloxp-division-badge eloxp-division-${divClass}`;
        divisionEl.textContent = division.toUpperCase();
      }

      // ELO display
      const eloDisplayEl = document.getElementById('eloxp-elo-display');
      if (eloDisplayEl) {
        eloDisplayEl.textContent = `${elo} ELO`;
      }

      // Progress bar
      const range = DIVISION_RANGES[division] || DIVISION_RANGES['rookie'];
      const progress = Math.min(100, Math.max(0, ((elo - range.min) / (range.max - range.min)) * 100));
      const progressFillEl = document.getElementById('eloxp-progress-fill');
      if (progressFillEl) {
        progressFillEl.style.width = `${progress}%`;
      }

      // ELO range
      const eloRangeEl = document.getElementById('eloxp-elo-range');
      if (eloRangeEl) {
        eloRangeEl.textContent = `${range.min} → ${range.max}`;
      }

      // Session record (W-L format instead of streak)
      const wins = data.hudData.sessionWins || data.hudData.winStreak || 0;
      const losses = data.hudData.sessionLosses || 0;
      const recordEl = document.getElementById('eloxp-session-record');
      if (recordEl) {
        recordEl.textContent = `${wins}W-${losses}L`;
      }

      // Session total XP
      const sessionTotalEl = document.getElementById('eloxp-session-total');
      if (sessionTotalEl && data.hudData.sessionEloxp !== undefined) {
        sessionTotalEl.textContent = data.hudData.sessionEloxp.toLocaleString();
      }
    }

    // Update move counter
    const moveEl = document.getElementById('eloxp-move');
    if (moveEl) {
      moveEl.textContent = `Round #${moveCount}`;
    }
  }

  function removeHUD() {
    if (hudElement) {
      hudElement.remove();
      hudElement = null;
    }
  }

  function showHUDMessage(message, type = 'info') {
    if (!hudElement) createHUD();
    
    const countEl = document.getElementById('eloxp-count');
    if (countEl && type === 'success') {
      countEl.classList.add('eloxp-success');
      setTimeout(() => countEl.classList.remove('eloxp-success'), 2000);
    }
  }

  // ============================================
  // Game Tracking
  // ============================================

  async function startTracking() {
    if (isTracking) return;

    const gameId = extractGameId();
    if (!gameId) return;

    const authResponse = await chrome.runtime.sendMessage({ type: 'CHECK_AUTH' });
    if (!authResponse.authenticated) {
      console.log('ELOXP: User not authenticated, skipping tracking');
      return;
    }

    currentGameId = gameId;
    userColor = getUserColor();
    opponentInfo = getOpponentInfo();
    isTracking = true;

    // Reset game state for new game
    sessionEloxp = 0;
    comboCount = 0;
    lastMultiplierCount = 0;
    shownMultipliers.clear();
    multiplierTotals = {}; // Reset tracked totals for new game
    moveCount = 0;
    lastPgn = '';
    if (comboTimer) {
      clearTimeout(comboTimer);
      comboTimer = null;
    }

    // Detect if playing against a bot
    isPracticeMode = opponentInfo?.isBot || false;
    
    console.log('ELOXP: Starting game tracking', { gameId, userColor, opponentInfo, isPracticeMode });

    // Create HUD first
    if (settings.hudEnabled) {
      createHUD();
    }
    
    // Set HUD status AFTER creating HUD (practice mode vs live)
    setHUDStatus(isPracticeMode ? 'practice' : 'live');
    
    // Hide any post-game actions from previous game
    hidePostGameActions();
    
    // Check if opponent is also on ELOXP (async, non-blocking)
    if (opponentInfo?.username && !opponentInfo.isBot) {
      lookupOpponentProfile(opponentInfo.username);
    }

    const gameData = {
      platform: 'chesscom',
      externalGameId: gameId,
      gameUrl: window.location.href,
      timeControl: getTimeControl(),
      userColor: userColor,
      opponentUsername: opponentInfo.username,
      opponentRating: opponentInfo.rating,
      isVsBot: opponentInfo.isBot, // Practice mode detection
      pgn: extractPgn(),
      currentMoveNumber: countMoves(extractPgn())
    };

    const response = await chrome.runtime.sendMessage({ 
      type: 'GAME_STARTED', 
      payload: gameData 
    });

    if (response.error) {
      console.error('ELOXP: Error starting game tracking:', response.error);
      if (response.error.includes('connect')) {
        showHUDMessage('Please connect your Chess.com account on ELOXP', 'warning');
      }
    } else {
      updateHUD(response);
    }

    observeMoves();
  }

  async function onMoveDetected() {
    if (!isTracking || !currentGameId) return;

    const pgn = extractPgn();
    const newMoveCount = countMoves(pgn);

    if (pgn !== lastPgn && newMoveCount > moveCount) {
      lastPgn = pgn;
      moveCount = newMoveCount;

      const gameData = {
        platform: 'chesscom',
        externalGameId: currentGameId,
        gameUrl: window.location.href,
        userColor: userColor,
        opponentUsername: opponentInfo?.username,
        opponentRating: opponentInfo?.rating,
        pgn: pgn,
        currentMoveNumber: newMoveCount
      };

      const response = await chrome.runtime.sendMessage({ 
        type: 'MOVE_MADE', 
        payload: gameData 
      });

      if (!response.error) {
        updateHUD(response);
      }
    }
  }

  async function onGameEnded() {
    if (!isTracking || !currentGameId) return;

    const result = getGameResult();
    if (!result) return;

    console.log('ELOXP: Game ended', result);

    const gameData = {
      platform: 'chesscom',
      externalGameId: currentGameId,
      pgn: extractPgn(),
      result: result.result,
      resultType: result.resultType,
      userColor: userColor
    };

    // Show "Calculating Performance..." animation
    showCalculatingOverlay();

    const response = await chrome.runtime.sendMessage({ 
      type: 'GAME_ENDED', 
      payload: gameData 
    });

    if (response.success) {
      // === DRAMATIC REVEAL SEQUENCE ===
      await showDramaticReveal(response, result);
      updateHUD(response);
      
      // Show post-game action buttons (Analyze with AI Coach)
      const gameId = currentGameId; // Capture before reset
      setTimeout(() => {
        showPostGameActions(gameId, result);
      }, 500);
    } else {
      hideCalculatingOverlay();
      console.error('ELOXP: Error completing game:', response.error);
      showHUDMessage('Sync failed - check connection', 'error');
    }

    isTracking = false;
    currentGameId = null;
  }

  // ============================================
  // DRAMATIC REVEAL ANIMATION
  // "Calculating Performance..." → ELOXP → ELO → Division
  // ============================================

  let calculatingOverlay = null;

  function showCalculatingOverlay() {
    if (calculatingOverlay) return;
    
    calculatingOverlay = document.createElement('div');
    calculatingOverlay.id = 'eloxp-calculating-overlay';
    calculatingOverlay.innerHTML = `
      <div class="eloxp-calc-content">
        <div class="eloxp-calc-spinner"></div>
        <div class="eloxp-calc-text">Calculating Performance...</div>
        <div class="eloxp-calc-subtext">Analyzing 47 skill multipliers</div>
      </div>
    `;
    document.body.appendChild(calculatingOverlay);
  }

  function hideCalculatingOverlay() {
    if (calculatingOverlay) {
      calculatingOverlay.remove();
      calculatingOverlay = null;
    }
  }

  async function showDramaticReveal(response, gameResult) {
    const earned = response.eloxpEarned || 0;
    const eloChange = response.eloChange || 0;
    const eloWeight = response.eloWeight || 0.5;
    const divisionProgress = response.divisionProgress || {};
    const divisionChanged = response.divisionChanged || false;
    const wasPromotion = response.wasPromotion || false;
    const profile = response.profile || {};
    
    // Create reveal overlay
    const revealOverlay = document.createElement('div');
    revealOverlay.id = 'eloxp-reveal-overlay';
    revealOverlay.innerHTML = `
      <div class="eloxp-reveal-content">
        <div class="eloxp-reveal-header">
          ${gameResult.result === 'win' ? '🏆 VICTORY!' : gameResult.result === 'draw' ? '🤝 DRAW' : '💪 GOOD FIGHT'}
        </div>
        
        <div class="eloxp-reveal-stages">
          <!-- Stage 1: ELOXP Earned -->
          <div class="eloxp-reveal-stage" id="stage-eloxp">
            <div class="eloxp-stage-label">SKILL POINTS EARNED</div>
            <div class="eloxp-stage-value eloxp-gold">+<span class="counter" data-target="${earned}">0</span> ELOXP</div>
          </div>
          
          <!-- Stage 2: ELO Change -->
          <div class="eloxp-reveal-stage hidden" id="stage-elo">
            <div class="eloxp-stage-label">ELO RATING <span class="eloxp-weight-badge">${Math.round(eloWeight * 100)}% PRACTICE</span></div>
            <div class="eloxp-stage-value ${eloChange >= 0 ? 'eloxp-green' : 'eloxp-red'}">
              ${eloChange >= 0 ? '+' : ''}<span class="counter" data-target="${eloChange}">0</span> ELO
            </div>
            <div class="eloxp-elo-result">${profile.previousEloRating || 0} → ${profile.eloRating || 0}</div>
          </div>
          
          <!-- Stage 3: Division Progress -->
          <div class="eloxp-reveal-stage hidden" id="stage-division">
            <div class="eloxp-stage-label">DIVISION PROGRESS</div>
            ${divisionChanged && wasPromotion ? `
              <div class="eloxp-promotion-banner">
                🎖️ PROMOTED TO ${(profile.division || 'contender').toUpperCase()}!
              </div>
            ` : divisionChanged ? `
              <div class="eloxp-demotion-banner">
                Now in ${(profile.division || 'rookie').toUpperCase()}
              </div>
            ` : `
              <div class="eloxp-progress-display">
                <div class="eloxp-progress-bar-mini">
                  <div class="eloxp-progress-fill-mini" style="width: ${divisionProgress.current || 0}%"></div>
                </div>
                <div class="eloxp-progress-text">
                  ${divisionProgress.eloToNextDivision ? `${divisionProgress.eloToNextDivision} ELO to ${divisionProgress.nextDivisionName || 'next division'}` : 'At highest division!'}
                </div>
              </div>
            `}
          </div>
        </div>
        
        <button class="eloxp-reveal-close" onclick="this.closest('#eloxp-reveal-overlay').remove()">
          Continue
        </button>
      </div>
    `;
    
    // Hide calculating, show reveal
    hideCalculatingOverlay();
    document.body.appendChild(revealOverlay);

    // Animate counter function
    function animateCounter(element, duration = 1000) {
      const target = parseInt(element.dataset.target) || 0;
      const start = 0;
      const startTime = performance.now();
      
      function update(currentTime) {
        const elapsed = currentTime - startTime;
        const progress = Math.min(elapsed / duration, 1);
        const easeProgress = 1 - Math.pow(1 - progress, 3);
        const current = Math.round(start + (target - start) * easeProgress);
        element.textContent = current;
        
        if (progress < 1) {
          requestAnimationFrame(update);
        }
      }
      requestAnimationFrame(update);
    }

    // Stage reveal sequence with delays
    await new Promise(resolve => setTimeout(resolve, 300));
    
    // Stage 1: ELOXP (already visible)
    const eloxpCounter = revealOverlay.querySelector('#stage-eloxp .counter');
    if (eloxpCounter) animateCounter(eloxpCounter, 800);
    
    await new Promise(resolve => setTimeout(resolve, 1200));
    
    // Stage 2: ELO
    const eloStage = revealOverlay.querySelector('#stage-elo');
    if (eloStage) {
      eloStage.classList.remove('hidden');
      eloStage.classList.add('reveal-animation');
      const eloCounter = eloStage.querySelector('.counter');
      if (eloCounter) animateCounter(eloCounter, 600);
    }
    
    await new Promise(resolve => setTimeout(resolve, 1000));
    
    // Stage 3: Division
    const divisionStage = revealOverlay.querySelector('#stage-division');
    if (divisionStage) {
      divisionStage.classList.remove('hidden');
      divisionStage.classList.add('reveal-animation');
    }
    
    // Auto-close after 8 seconds if not dismissed
    setTimeout(() => {
      const overlay = document.getElementById('eloxp-reveal-overlay');
      if (overlay) overlay.remove();
    }, 8000);
  }

  // ============================================
  // DOM Observation
  // ============================================

  let moveObserver = null;
  let gameEndObserver = null;

  function observeMoves() {
    const moveList = document.querySelector('.move-list, .vertical-move-list, .move-list-wrapper');
    if (!moveList) {
      setTimeout(observeMoves, 1000);
      return;
    }

    moveObserver = new MutationObserver((mutations) => {
      clearTimeout(moveObserver.timeout);
      moveObserver.timeout = setTimeout(onMoveDetected, 200);
    });

    moveObserver.observe(moveList, {
      childList: true,
      subtree: true,
      characterData: true
    });

    gameEndObserver = new MutationObserver((mutations) => {
      const result = getGameResult();
      if (result) {
        onGameEnded();
        gameEndObserver.disconnect();
      }
    });

    gameEndObserver.observe(document.body, {
      childList: true,
      subtree: true
    });
  }

  function stopObserving() {
    if (moveObserver) {
      moveObserver.disconnect();
      moveObserver = null;
    }
    if (gameEndObserver) {
      gameEndObserver.disconnect();
      gameEndObserver = null;
    }
  }

  // ============================================
  // Page Navigation Detection
  // ============================================

  let lastUrl = window.location.href;

  function checkForPageChange() {
    if (window.location.href !== lastUrl) {
      lastUrl = window.location.href;
      
      stopObserving();
      isTracking = false;
      currentGameId = null;
      lastPgn = '';
      moveCount = 0;
      removeHUD();

      if (isGamePage() && isActiveGame()) {
        setTimeout(startTracking, 500);
      }
    }
  }

  // ============================================
  // Message Handler
  // ============================================

  chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.type === 'SETTINGS_UPDATED') {
      settings = message.payload;
      if (!settings.hudEnabled) {
        removeHUD();
      } else if (isTracking) {
        createHUD();
      }
    }
    if (message.type === 'ELOXP_UPDATE') {
      updateHUD(message.payload);
    }
    // Keyboard shortcut: Ctrl+Shift+E to toggle HUD
    if (message.type === 'TOGGLE_HUD') {
      toggleHUD();
    }
  });

  // Toggle HUD visibility (keyboard shortcut handler)
  function toggleHUD() {
    if (!hudElement) {
      // Create HUD if it doesn't exist and we're authenticated
      chrome.runtime.sendMessage({ type: 'CHECK_AUTH' }).then(response => {
        if (response?.authenticated) {
          createHUD();
          showHUDToast('HUD Enabled (Ctrl+Shift+E to toggle)');
        }
      });
    } else {
      // Toggle visibility
      const isHidden = hudElement.style.display === 'none';
      hudElement.style.display = isHidden ? 'block' : 'none';
      if (isHidden) {
        showHUDToast('HUD Visible');
      }
    }
  }

  // Show a brief toast notification near the HUD
  function showHUDToast(message) {
    const toast = document.createElement('div');
    toast.className = 'eloxp-toast';
    toast.textContent = message;
    document.body.appendChild(toast);
    setTimeout(() => {
      toast.classList.add('eloxp-toast-exit');
      setTimeout(() => toast.remove(), 300);
    }, 2000);
  }

  // ============================================
  // Opponent ELOXP Profile Lookup
  // ============================================

  let opponentEloxpProfile = null;

  async function lookupOpponentProfile(username) {
    try {
      const response = await chrome.runtime.sendMessage({
        type: 'LOOKUP_OPPONENT',
        payload: { platform: 'chesscom', username }
      });
      
      if (response && response.found && response.profile) {
        opponentEloxpProfile = response.profile;
        console.log('ELOXP: Opponent is an ELOXP fighter!', opponentEloxpProfile);
        showOpponentBadge(opponentEloxpProfile);
      } else {
        opponentEloxpProfile = null;
        console.log('ELOXP: Opponent not on ELOXP platform');
      }
    } catch (e) {
      console.log('ELOXP: Error looking up opponent:', e);
      opponentEloxpProfile = null;
    }
  }

  function showOpponentBadge(profile) {
    const existingBadge = document.getElementById('eloxp-opponent-badge');
    if (existingBadge) existingBadge.remove();
    
    const hudBody = document.getElementById('eloxp-hud-body');
    if (!hudBody) return;
    
    const division = (profile.division || 'rookie').toLowerCase();
    const divClass = getDivisionClass(division);
    
    const badgeDiv = document.createElement('div');
    badgeDiv.id = 'eloxp-opponent-badge';
    badgeDiv.className = 'eloxp-opponent-section';
    badgeDiv.innerHTML = `
      <div class="eloxp-opponent-header">
        <span class="eloxp-opponent-label">⚔️ VS ELOXP FIGHTER</span>
      </div>
      <div class="eloxp-opponent-info">
        <span class="eloxp-division-badge eloxp-division-${divClass}">${division.toUpperCase()}</span>
        <span class="eloxp-opponent-name">${profile.displayName || profile.username}</span>
        <span class="eloxp-opponent-elo">${profile.eloRating || 800} ELO</span>
      </div>
      <a href="https://eloxp.com/fighter/${encodeURIComponent(profile.username)}" 
         target="_blank" 
         class="eloxp-view-profile-link">
        View Profile →
      </a>
    `;
    
    const divisionSection = hudBody.querySelector('.eloxp-division-section');
    if (divisionSection) {
      divisionSection.after(badgeDiv);
    } else {
      hudBody.prepend(badgeDiv);
    }
    
    playSound('ding');
    showHUDToast(`⚔️ Opponent is an ELOXP Fighter!`);
  }

  function hideOpponentBadge() {
    const badge = document.getElementById('eloxp-opponent-badge');
    if (badge) badge.remove();
    opponentEloxpProfile = null;
  }

  // ============================================
  // Post-Game Actions
  // ============================================

  let lastCompletedGameId = null;

  function showPostGameActions(gameId, gameResult) {
    lastCompletedGameId = gameId;
    setHUDStatus('game_over');
    
    const actionsContainer = document.getElementById('eloxp-post-game-actions');
    if (!actionsContainer) {
      const hudBody = document.getElementById('eloxp-hud-body');
      if (hudBody) {
        const postGameDiv = document.createElement('div');
        postGameDiv.id = 'eloxp-post-game-actions';
        postGameDiv.className = 'eloxp-post-game-section';
        postGameDiv.innerHTML = `
          <a href="https://eloxp.com/analysis?source=chesscom&gameId=${gameId}" 
             target="_blank" 
             class="eloxp-analyze-btn">
            🧠 Analyze with AI Coach
          </a>
          <a href="https://eloxp.com/dashboard" 
             target="_blank" 
             class="eloxp-dashboard-btn">
            📊 View Dashboard
          </a>
        `;
        hudBody.appendChild(postGameDiv);
      }
    } else {
      const analyzeBtn = actionsContainer.querySelector('.eloxp-analyze-btn');
      if (analyzeBtn) {
        analyzeBtn.href = `https://eloxp.com/analysis?source=chesscom&gameId=${gameId}`;
      }
      actionsContainer.style.display = 'block';
    }
  }

  function hidePostGameActions() {
    const actionsContainer = document.getElementById('eloxp-post-game-actions');
    if (actionsContainer) {
      actionsContainer.style.display = 'none';
    }
  }

  // ============================================
  // Initialization
  // ============================================

  async function init() {
    console.log('ELOXP: Chess.com content script loaded');

    const settingsResponse = await chrome.runtime.sendMessage({ type: 'GET_SETTINGS' });
    if (settingsResponse) {
      settings = settingsResponse;
    }

    if (isGamePage()) {
      if (document.readyState === 'complete') {
        if (isActiveGame()) {
          startTracking();
        }
      } else {
        window.addEventListener('load', () => {
          if (isActiveGame()) {
            startTracking();
          }
        });
      }
    }

    setInterval(checkForPageChange, 500);
  }

  init();
})();
